/*
 * $Id:  $
 *
 * This file is part of the jcar (R) project.
 * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
 * Authors: laurent.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or(at your option)any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>
 */
package com.indigosoftware.idp.component.agent.dsignature.engine;


import com.itextpdf.awt.geom.Rectangle2D;
import com.itextpdf.text.pdf.*;
import com.itextpdf.text.Image;
import com.itextpdf.text.pdf.parser.PdfReaderContentParser;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import java.io.File;
import java.util.concurrent.ConcurrentHashMap;
import java.util.zip.CRC32;


public class SealPositionService {
    private static Logger logger = Logger.getLogger(SealPositionService.class);

    private String sealId;

    private byte[] sealData;

    private ConcurrentHashMap<String, SealPosition> sealPositions = new ConcurrentHashMap<String, SealPosition>();

    public SealPosition getSealPosition(String seal, PdfReader reader) throws Exception {
        sealData = FileUtils.readFileToByteArray(new File(seal));
        CRC32 crc32 = new CRC32();
        crc32.update(sealData);
        sealId = Long.toHexString(crc32.getValue());

        if(!sealPositions.containsKey(seal)){

            String positionFile = seal.replace(".png", ".xml");
            if(!new File(positionFile).isFile()){
                throw new Exception("Seal position file not found: " + positionFile);
            }
            SealPosition sealPosition = new SealPosition();
            sealPosition.update(positionFile);
            sealPositions.putIfAbsent(seal, sealPosition);
        }
        SealPosition sealPosition = sealPositions.get(seal);
        if(StringUtils.isNotBlank(sealPosition.getKey())){
            sealPosition = new SealPosition(sealPosition);
            updatePositionByKey(sealPosition, reader);
        }
        return sealPosition;
    }

    private void updatePositionByKey(SealPosition sealPosition, PdfReader reader) throws Exception {
        if(sealPosition.getPage() >= reader.getNumberOfPages()){
           sealPosition.setPage(reader.getNumberOfPages());
        }

      
        PdfReaderContentParser parser = new PdfReaderContentParser(reader);
        int totalPages = reader.getNumberOfPages();

        if(sealPosition.getPage() > 0){
            KeyWordPositionListener listener = new KeyWordPositionListener();
            listener.setKeyWord(sealPosition.getKey());

            parser.processContent(sealPosition.getPage(), listener);

            if (listener.getFoundPosition() != null) {
                update(listener.getFoundPosition(), sealPosition);
            }
        }else {
            for (int pageNum = 1; pageNum <= totalPages; pageNum++) {
                KeyWordPositionListener listener = new KeyWordPositionListener();
                listener.setKeyWord(sealPosition.getKey());

                parser.processContent(pageNum, listener);

                if (listener.getFoundPosition() != null) {
                    sealPosition.setPage(pageNum);
                    update(listener.getFoundPosition(), sealPosition);
                    break;
                }
            }
        }
    }


    /**
     * 根据关键字定位印章位置, x，y为毫米单位
     * 0为左下角对齐，1为居中对齐，2为右下角对齐
     * @param textRectangle
     * @param sealPosition
     */
    private void update(Rectangle2D.Float textRectangle, SealPosition sealPosition) {
        float x = textRectangle.x * SealPosition.POINT_TO_MM;
        float y = textRectangle.y * SealPosition.POINT_TO_MM;

        switch (sealPosition.getAlign()) {
            case 0: // 左下角对齐
                break;
            case 1: // 居中对齐
                x = textRectangle.x * SealPosition.POINT_TO_MM + (textRectangle.width * SealPosition.POINT_TO_MM - sealPosition.getWidth()) / 2;
                y = textRectangle.y * SealPosition.POINT_TO_MM + (textRectangle.height * SealPosition.POINT_TO_MM - sealPosition.getHeight()) / 2;
                break;
            case 2: // 右下角对齐
                x = textRectangle.x * SealPosition.POINT_TO_MM + textRectangle.width * SealPosition.POINT_TO_MM - sealPosition.getWidth();
                y = textRectangle.y * SealPosition.POINT_TO_MM;
                break;
        }
        sealPosition.setX(x);
        sealPosition.setY(y);
    }

    public void drawSignature(int page, PdfTemplate content, int dpi, float offsetX, float offsetY) throws Exception {

        try {
                content.saveState();


                logger.debug("drawSignature seal: " + this.getClass().getName() + " , pageNo : " + page);


                Image image = Image.getInstance(sealData);
                this.addSignAttribute(image, 1, sealId);


                image.scalePercent(72.0F / (float)dpi * 100.0F);
                image.setAbsolutePosition(offsetX, offsetY);
                content.addImage(image);
                image.setAbsolutePosition(-offsetX, -offsetY);

                logger.debug("seal complete");

        } catch (Exception e) {
            logger.error("PdfContentByte add seal image Exception, ", e);
            throw e;
        } finally {
            content.restoreState();
        }
    }

    protected void addSignAttribute(Image image, int value, String sealId) {
        PdfDictionary imageDict = image.getAdditional();
        if (imageDict == null) {
            imageDict = new PdfDictionary();
            image.setAdditional(imageDict);
        }

        imageDict.put(new PdfName("Sign"), new PdfNumber(value));
        if (sealId != null) {
            imageDict.put(new PdfName("SealID"), new PdfName(sealId));
        }

    }
}
