/*
 * $Id:  $
 *
 * This file is part of the jcar (R) project.
 * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
 * Authors: laurent.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or(at your option)any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>
 */
package com.indigosoftware.idp.component.agent.dsignature.engine;

import com.indigopacific.module.model.Signature;
import com.indigosoftware.adapter.jce.common.SignatureDevice;
import com.indigosoftware.idp.component.agent.dsignature.engine.pdf.MakeSignature;
import com.indigosoftware.idp.component.agent.dsignature.factory.WorkMode;
import com.indigosoftware.idp.component.agent.dsignature.signaturedevices.LocalSignatureDeviceManager;
import com.indigosoftware.idp.component.agent.dsignature.signaturedevices.SignatureDeviceManager;
import com.indigosoftware.idp.component.agent.dsignature.util.DigitalAssist;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.*;
import com.itextpdf.text.pdf.security.CertificateInfo;
import com.itextpdf.text.pdf.security.ExternalDigest;
import com.itextpdf.text.pdf.security.ExternalSignature;
import com.itextpdf.text.pdf.security.MakeSignature.CryptoStandard;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.dom4j.Element;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.security.GeneralSecurityException;
import java.security.MessageDigest;
import java.security.cert.Certificate;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.text.SimpleDateFormat;
import java.util.*;


public class PDFSealSignatureEngine implements DigitalSignatureEngine {

	private static Logger logger = Logger.getLogger(PDFSealSignatureEngine.class);
	
	protected String hashAlgorithm = "SHA1";
	
	protected String encryptionAlgorithm = "RSA";
	
	protected String deviceName = null;
	
    private Map<String, Object> parameters;
    
    public static ThreadLocal<String>  encryptionAlgorithmTL = new ThreadLocal<String>();


	@Override
	public byte[] sign(byte[] bytes, SignatureDeviceManager manager,
					   Signature signature) throws Exception {
		logger.info("PDFSignatureEngine start to sign", null);
		PdfReader reader = null;
		PdfStamper stamper = null;
		ByteArrayOutputStream bos = null;
		try {
			reader = new PdfReader(bytes);
			if(parameters.containsKey("PDFAllow")){
				bos = new ByteArrayOutputStream();
				stamper = new PdfStamper(reader, bos);
				int allow = 0;
			
				allow = Integer.parseInt((String) parameters.get("PDFAllow"));
				if(parameters.get("PDFOwnerPWD") == null){
					throw new Exception( "The PDFOwnerPWD must be supplied when Controlling PDFAllow");
				}
				String owner = (String) parameters.get("PDFOwnerPWD");
				String user = null;
				if(parameters.get("PDFUserPWD") != null){
					user = parameters.get("PDFUserPWD").toString();
				}
				stamper.setEncryption(user == null? null : user.getBytes(), owner.getBytes(), allow, PdfWriter.ENCRYPTION_AES_128 | PdfWriter.DO_NOT_ENCRYPT_METADATA);
				stamper.close();
				reader = new PdfReader(bos.toByteArray(), owner.getBytes());
			}
			
			
			bos = new ByteArrayOutputStream();
			stamper = PdfStamper.createSignature(reader, bos, '\0', null, true);
			SignatureDevice searchDevice = null;
			if(manager instanceof LocalSignatureDeviceManager){
				searchDevice = manager.getDevice(signature.getDeviceName(), signature.getDeviceType());

				Map<String, String> env = new HashMap<String, String>();
				for (Object ele : signature.getDeviceEle().elements()) {
					env.put(((Element)ele).getName(), ((Element)ele).getTextTrim());
				}
				searchDevice.setAlgorithm(hashAlgorithm, encryptionAlgorithm);
				searchDevice.initialize(env);
			}
			logger.debug( "PDFSignatureEngine getDevice successful", null);
			final SignatureDevice device = searchDevice;
			
			device.setAlgorithm(hashAlgorithm, encryptionAlgorithm);
			PdfSignatureAppearance appearance = stamper.getSignatureAppearance();
			logger.debug("PDFSignatureEngine getAppearance successful", null);
			
			appearance.setCertificationLevel(signature.getCertifyLevel());
			
			Certificate[] chain = null;
			//由设备提供证书
			chain = device.getChain();

			if (chain == null) {
				throw new Exception( "The certificate must be supplied when signing a normal signature");
			}
			//检查证书有效期
			Method checkValidM = chain[0].getClass().getDeclaredMethod("checkValidity");
			try{
				checkValidM.invoke(chain[0]);
			}catch(InvocationTargetException e){
				if(e.getTargetException() != null && (e.getTargetException() instanceof CertificateExpiredException
						|| e.getTargetException() instanceof CertificateNotYetValidException)){
					    X509Certificate certificate = (X509Certificate)chain[0];
					    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss,SSS");
						Map<String, String> messages = new HashMap<String, String>();
						messages.put("SN", certificate.getSerialNumber().toString());
						messages.put("DN", certificate.getSubjectDN().toString());
						messages.put("NotBefore()", format.format(certificate.getNotBefore()));
						messages.put("NotAfter", format.format(certificate.getNotAfter()));
						messages.put("Exception", e.getTargetException().getClass().getSimpleName());
//						ResourceAdapterFactory.getResourceAdapter(null).report(MonitorType.Cert, messages);
				}
				throw new Exception("checkValidity failed:[{errorMessage}] ", e);
			}
			appearance.setReason(signature.getReason());
			if((signature.getLocation() == null ||signature.getLocation().equals(""))
					&& chain[0] instanceof X509Certificate){
				appearance.setLocation(CertificateInfo.getSubjectFields((X509Certificate) chain[0]).getField("CN"));
			}else{
				appearance.setLocation(signature.getLocation());
			}
			appearance.setContact(signature.getContact());
			
//			logger.debug("", "PDFSignatureEngine set location contact successful", null);
			long time1 = System.currentTimeMillis();
			ArrayList blankSignatureFields = reader.getAcroFields().getBlankSignatureNames();//获得空白签名域
			
			long time2 = System.currentTimeMillis();
//			logger.debug("", "1[{time}] size[{size}]", new String[]{time2-time1+"", blankSignatureFields.size()+""});
			ArrayList signatureFields = reader.getAcroFields().getSignatureNames();//获得已经存在签名或被签名的域
			
			long time3 = System.currentTimeMillis();
//			logger.debug("", "2[{time}] size[{size}]", new String[]{time3-time2+"", signatureFields.size()+""});
			signatureFields.removeAll(blankSignatureFields);
//			logger.debug("", "3", null);
			if (signatureFields.contains(signature.getSignatureField())) {
//				logger.debug("", "4", null);
				signature.setSignatureField(DigitalAssist.createSignatureFieldName(signature.getIdentify()));
//				logger.debug("", "5", null);
			}

			SealPositionService sealPositionService = new SealPositionService();
			SealPosition sealPosition = sealPositionService.getSealPosition((String) this.parameters.get("seal"), reader);


			float x = sealPosition.getX() * SealPosition.mm2inch * SealPosition.dpi_default;
			float y = sealPosition.getY() * SealPosition.mm2inch * SealPosition.dpi_default;
			float imgh = SealPosition.dpi_default * sealPosition.getHeight() * SealPosition.mm2inch;

			float imgw = SealPosition.dpi_default * sealPosition.getWidth() * SealPosition.mm2inch;
			float urx = x + imgw;
			float ury = y + imgh;

			appearance.setVisibleSignature(new Rectangle(x, y, urx + 1.1f, ury + 1.1f),
					sealPosition.getPage(), signature.getSignatureField());
			//盖章
			sealPositionService.drawSignature(sealPosition.getPage(), appearance.getTopLayer(), sealPosition.getDpi(),
					sealPosition.getOffsetX() * SealPosition.mm2inch * SealPosition.dpi_default,
					sealPosition.getOffsetY() * SealPosition.mm2inch * SealPosition.dpi_default);


			logger.debug("PDFSignatureEngine setVisibleSignature successful", null);
			ExternalDigest digest = new ExternalDigest() {
				
				@Override
				public MessageDigest getMessageDigest(String hashAlgorithm)
						throws GeneralSecurityException {
//					MessageDigest md = null;
					try{
					
						return device.getMessageDigest(hashAlgorithm);
						
					}catch(Exception e){
					   throw new GeneralSecurityException(e);
					}
//					return md;
				}
			};
//			
			ExternalSignature pks = new ExternalSignature() {
				
				@Override
				public byte[] sign(byte[] paramArrayOfByte) throws GeneralSecurityException {
					try {
						return device.sign(paramArrayOfByte);
					} catch (Exception e) {
						throw new GeneralSecurityException(e);
					}
				}
				
				@Override
				public String getHashAlgorithm() {
					return device.getHashAlgorithm();
				}
				
				@Override
				public String getEncryptionAlgorithm() {
					return device.getEncryptionAlgorithm();
				}
			};

			logger.debug( "PDFSignatureEngine signDetached", null);
			
			encryptionAlgorithmTL.set(encryptionAlgorithm);
			MakeSignature.signDetached(appearance, digest, pks, chain, 
				 null, null, null, 0, CryptoStandard.CMS, device); 
			encryptionAlgorithmTL.remove();
			logger.debug( "PDFSignatureEngine signDetached successful", null);

		} catch (Exception e) {
			//签名{signatrue}处理出现异常，错误原因为{CatchedException}”。
			throw new Exception("PDFSignature sign signature[{signatrue}] failed, errorMessage = [{message}]", e);
		} finally {
			try {
				if (reader != null) {
					reader.close();
				}
				if (bos != null) {
					bos.close();
				}
			} catch (Exception e) {
				throw new Exception("close reader / bos failed", e);
			}
		}
		
		return bos.toByteArray();
	}


    @Override
	public void getRevision(String in, String out, int revision) throws Exception {
		PdfReader reader = null;
		try {
			reader = new PdfReader(in);
			AcroFields acroFields = reader.getAcroFields();
			List<String> signatureNames = acroFields.getSignatureNames();
			
			if(signatureNames == null || signatureNames.size() == 0){
				throw new Exception("there has no signature in IN pdf");
			}
		
			if(signatureNames.size() < revision){
	    		throw new Exception("getRevision failed, num of signatureRevisions[{size}] is less than parameter revision[{revision}]");
			}
			for (Iterator<String> it = signatureNames.iterator(); it.hasNext();) {
				String signatureName =  it.next();
				int currentRevision = acroFields.getRevision(signatureName);
				if(currentRevision == revision){
					InputStream is = acroFields.extractRevision(signatureName);
					File file = new File(out);
					FileOutputStream fos = new FileOutputStream(file);
					IOUtils.copy(is, fos);
					IOUtils.closeQuietly(fos);
				}
			}
    	}catch(Exception e){
    		throw e;
    	}
	}


	@Override
	public void setParameters(Map<String, Object> parameters) throws Exception {
        this.parameters = parameters;
        if(parameters.get("digestType") != null){
        	this.hashAlgorithm = (String) parameters.get("digestType");
        }
        if(parameters.get("signType") != null){
        	this.encryptionAlgorithm = (String) parameters.get("signType");
        }
        if(parameters.get("deviceName") != null){
        	this.deviceName = (String) parameters.get("deviceName");
        }else{
        	if(WorkMode.getValue((String) parameters.get("workMode")) == WorkMode.SIGN){
        		if(parameters.get("isLocal") != null && parameters.get("isLocal").equals(true) ){
        			
        		}else{
                	throw new Exception("The parameter deviceName must be supplied");
        		}
        	}
        }
    }



	
}
