/*
 * $Id:  $
 *
 * This file is part of the jcar (R) project.
 * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
 * Authors: laurent.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or(at your option)any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>
 */
package com.indigosoftware.idp.component.agent.dsignature.engine;

import com.itextpdf.awt.geom.Rectangle2D;
import com.itextpdf.text.pdf.parser.*;

import java.util.ArrayList;
import java.util.List;

public class KeyWordPositionListener implements RenderListener {
    
    private String keyWord;
    private Rectangle2D.Float foundPosition;

    
    // 用于文本重组的状态变量
    private StringBuilder currentText = new StringBuilder();

    private Rectangle2D.Float currentTextBoundingBox;

    private TextRenderInfo lastRenderInfo;

    private List<TextRenderInfo> textRenderInfos = new ArrayList<>();


    public void setKeyWord(String keyWord) {
        this.keyWord = keyWord;
    }
    
    public Rectangle2D.Float getFoundPosition() {
        return foundPosition;
    }



    @Override
    public void beginTextBlock() {
        // 开始新的文本块时重置状态
        currentText.setLength(0);
        currentTextBoundingBox = null;
        textRenderInfos.clear();
    }

    @Override
    public void renderText(TextRenderInfo renderInfo) {
        if (keyWord == null || foundPosition != null) {
            return; // 已找到关键词或未设置关键词，直接返回
        }

        String textFragment = renderInfo.getText();
        Rectangle2D.Float textRectangle = renderInfo.getBaseline().getBoundingRectange();

        // 判断是否与上一个文本片段在同一行（Y坐标相近）
        boolean isSameLine = lastRenderInfo != null && 
                           Math.abs(lastRenderInfo.getBaseline().getBoundingRectange().y - textRectangle.y) < 5;

        if (isSameLine && isAdjacentText(lastRenderInfo, renderInfo)) {
            // 同一行且相邻的文本，进行拼接
            currentText.append(textFragment);
            // 合并边界框
            mergeBoundingBoxes(renderInfo);
        } else {
            // 新行或非相邻文本，开始新的拼接
            checkCurrentTextForKeyword();
            
            currentText.setLength(0);
            currentText.append(textFragment);
            currentTextBoundingBox = new Rectangle2D.Float(
                textRectangle.x, textRectangle.y, textRectangle.width, textRectangle.height
            );
        }

        textRenderInfos.add(renderInfo);
        lastRenderInfo = renderInfo;
    }

    @Override
    public void endTextBlock() {
        // 文本块结束时，检查最后拼接的文本
        checkCurrentTextForKeyword();
    }

    @Override
    public void renderImage(ImageRenderInfo renderInfo) {
        // 不处理图片
    }

    /**
     * 判断两个文本片段是否相邻
     */
    private boolean isAdjacentText(TextRenderInfo previous, TextRenderInfo current) {
        if (previous == null) return false;
        
        Rectangle2D.Float prevRect = previous.getBaseline().getBoundingRectange();
        Rectangle2D.Float currRect = current.getBaseline().getBoundingRectange();
        
        float prevEndX = prevRect.x + prevRect.width;
        float distance = currRect.x - prevEndX;
        
        // 判断X坐标是否接近（考虑字体大小）
        float avgFontSize = (prevRect.height + currRect.height) / 2;
        return distance <= avgFontSize * 0.5f; // 距离小于字体大小的50%视为相邻
    }

    /**
     * 合并文本片段的边界框
     */
    private void mergeBoundingBoxes(TextRenderInfo newText) {
        if (currentTextBoundingBox == null) {
            return;
        }
        
        Rectangle2D.Float newRect = newText.getBaseline().getBoundingRectange();
        float newLeft = Math.min(currentTextBoundingBox.x, newRect.x);
        float newRight = Math.max(currentTextBoundingBox.x + currentTextBoundingBox.width, newRect.x + newRect.width);

        currentTextBoundingBox.x = newLeft;
        currentTextBoundingBox.width = newRight - newLeft;
        // 根据需要调整Y坐标和高度
        currentTextBoundingBox.y = Math.min(currentTextBoundingBox.y, newRect.y);
        currentTextBoundingBox.height = Math.max(currentTextBoundingBox.height, newRect.height);
    }

    /**
     * 检查当前拼接的文本是否包含关键词
     */
    private void checkCurrentTextForKeyword() {
        if (currentText.length() == 0 || keyWord == null) {
            return;
        }

        String assembledText = currentText.toString();
        int keywordIndex = assembledText.indexOf(keyWord);
        if (keywordIndex >= 0) {
            foundPosition = calculateKeywordBoundingBox(keywordIndex, keyWord.length());
        }
    }

    /**
     * 根据关键词在文本中的位置，计算其精确的边界框
     * @param startIndex 关键词在拼接文本中的起始位置
     * @param length 关键词的长度
     * @return 关键词的精确边界框
     */
    private Rectangle2D.Float calculateKeywordBoundingBox(int startIndex, int length) {
        if (textRenderInfos.isEmpty() || length <= 0) {
            return currentTextBoundingBox;
        }


        float minX = Float.MAX_VALUE;
        float minY = Float.MAX_VALUE;
        float maxX = Float.MIN_VALUE;
        float maxY = Float.MIN_VALUE;

        int currentCharIndex = 0;
        for (TextRenderInfo renderInfo : textRenderInfos) {
            String text = renderInfo.getText();
            for (int i = 0; i < text.length(); i++) {
                if (currentCharIndex >= startIndex && currentCharIndex < startIndex + length) {
                    // 这个字符属于关键词，将其边界框加入计算
                    Rectangle2D.Float charRect = getCharacterBoundingBox(renderInfo, i);
                    minX = Math.min(minX, charRect.x);
                    minY = Math.min(minY, charRect.y);
                    maxX = Math.max(maxX, charRect.x + charRect.width);
                    maxY = Math.max(maxY, charRect.y + charRect.height);
                }
                currentCharIndex++;
            }
        }

        if (minX != Float.MAX_VALUE) {
            return new Rectangle2D.Float(minX, minY, maxX - minX, maxY - minY);
        } else {
            return currentTextBoundingBox;
        }
    }

    /**
     * 获取单个字符的边界框
     * @param renderInfo 文本渲染信息
     * @param charIndex 字符在文本中的索引
     * @return 字符的边界框
     */
    private Rectangle2D.Float getCharacterBoundingBox(TextRenderInfo renderInfo, int charIndex) {
        Rectangle2D.Float baseRect = renderInfo.getBaseline().getBoundingRectange();

        if (renderInfo.getText().length() > 0) {
            float avgCharWidth = baseRect.width / renderInfo.getText().length();
            float charX = baseRect.x + charIndex * avgCharWidth;
            return new Rectangle2D.Float(charX, baseRect.y, avgCharWidth, baseRect.height);
        } else {
            return baseRect;
        }
    }
}