/*
 * $Id:  $
 *
 * This file is part of the jcar (R) project.
 * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
 * Authors: laurent.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or(at your option)any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>
 */
package com.indigosoftware.adapter.jce.common;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CRL;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;


public abstract class SignatureDevice {
    private static Logger logger = Logger.getLogger(SignatureDevice.class);

    protected String hashAlgorithm;

    protected String encryptionAlgorithm;

    protected String provider;

    protected boolean initialized = false;

    protected Map<String, String> parameters = new HashMap<String, String>();

    protected Certificate[] chain;

    protected KeyStore keyStore;

    protected String trustCert;

    public abstract void initialize(Map<String, String> paramMap) throws Exception;

    public abstract void destroy();

    public abstract byte[] sign(byte[] paramArrayOfbyte) throws Exception;

    public MessageDigest getMessageDigest(String hashAlgorithm) throws Exception {
        MessageDigest md = null;
        try {
            if (getProvider() == null) {
                md = MessageDigest.getInstance(hashAlgorithm);
            } else {
                md = MessageDigest.getInstance(hashAlgorithm, getProvider());
            }
        } catch (Exception e) {
            logger.warn(e);
            md = MessageDigest.getInstance(getHashAlgorithm(), "Sun");
        }
        return md;
    }

    public Signature getSignature(String algorithm) throws Exception {
        Signature sin = null;
        try {
            if (getProvider() == null) {
                sin = Signature.getInstance(algorithm);
            } else {
                sin = Signature.getInstance(algorithm, getProvider());
            }
        } catch (Exception e) {
            logger.warn(e);
            sin = Signature.getInstance(algorithm, "Sun");
        }
        return sin;
    }

    public abstract Object verify(byte[] paramArrayOfbyte) throws Exception;

    public Certificate[] getChain() {
        return this.chain;
    }

    protected void setChain(String certPath) throws Exception {
        FileInputStream fis = null;
        try {
            fis = new FileInputStream(certPath);
            CertificateFactory cf = CertificateFactory.getInstance("X509");
            Collection<Certificate> col = (Collection)cf.generateCertificates(fis);
            this.chain = new Certificate[col.size()];
            int i = 0;
            for (Iterator<Certificate> it = col.iterator(); it.hasNext(); ) {
                X509Certificate cert = (X509Certificate)it.next();
                this.chain[i++] = cert;
            }
        } catch (FileNotFoundException e) {
            throw e;
        } catch (CertificateException e) {
            throw e;
        } finally {
            try {
                if (fis != null) {
                    fis.close();
                }
            } catch (IOException e) {
                fis = null;
            }
        }
    }

    public void setAlgorithm(String hashAlgorithm, String encryptionAlgorithm) {
        this.hashAlgorithm = hashAlgorithm;
        this.encryptionAlgorithm = encryptionAlgorithm;
    }

    public String getHashAlgorithm() {
        return this.hashAlgorithm;
    }

    public String getEncryptionAlgorithm() {
        return this.encryptionAlgorithm;
    }

    public Map<String, String> getParameters() {
        return this.parameters;
    }

    public String getProvider() {
        return this.provider;
    }

    public boolean isInitialized() {
        return this.initialized;
    }

    public void setInitialized(boolean initialized) {
        this.initialized = initialized;
    }

    public boolean verifiyCertificate(X509Certificate cert, Collection<CRL> crls, Calendar calendar) {
        if (this.trustCert != null) {
            logger.debug("verifiyCertificateByTrustCert:" + this.trustCert, null);
            return verifiyCertificateByTrustCert(cert, crls, calendar);
        }
        logger.debug("verifiyCertificateByKeyStore", null);
        return verifiyCertificateByKeyStore(cert, crls, calendar);
    }

    public boolean verifiyCertificateByTrustCert(X509Certificate cert, Collection<CRL> crls, Calendar calendar) {
        try {
            List<X509Certificate> list = genX509Certificates(this.trustCert);
            for (X509Certificate certStoreX509 : list) {
                try {
                    if (verifyCert(certStoreX509, crls, calendar) != null) {
                        continue;
                    }
                    try {
                        logger.debug("verify publicKey:", null);
                        if (this.provider != null) {
                            cert.verify(certStoreX509.getPublicKey(), this.provider);
                        } else {
                            try {
                                cert.verify(certStoreX509.getPublicKey(), "BC");
                            } catch (Exception e) {
                                cert.verify(certStoreX509.getPublicKey());
                            }
                        }
                        return true;
                    } catch (Exception e) {
                        logger.debug( "spc.getSubjectDN:" + certStoreX509.getSubjectDN(), null);
                    }
                } catch (Exception ex) {}
            }
        } catch (Exception e) {}
        return false;
    }

    public List<X509Certificate> genX509Certificates(String chain) throws Exception {
        FileInputStream fis = null;
        List<X509Certificate> list = new ArrayList<X509Certificate>();
        try {
            File chainFile = new File(chain);
            if (!chainFile.exists()) {
                throw new Exception("Certificate chain file doesn't exist:" + chain);
            }
            fis = new FileInputStream(chain);
            CertificateFactory cf = null;
            if (this.provider != null) {
                cf = CertificateFactory.getInstance("X509", this.provider);
            } else {
                cf = CertificateFactory.getInstance("X509", "BC");
            }
            X509Certificate certTemp = (X509Certificate)cf.generateCertificate(fis);
            fis.close();
            fis = new FileInputStream(chain);
            if (certTemp.getSigAlgName().toUpperCase().startsWith("SM")) {
                cf = CertificateFactory.getInstance("X509", this.provider);
            } else {
                cf = CertificateFactory.getInstance("X509");
            }
            Exception error = null;
            Collection<? extends Certificate> col = null;
            try {
                col = cf.generateCertificates(fis);
            } catch (CertificateException e) {
                error = e;
            }
            try {
                if (col == null) {
                    fis.close();
                    cf = CertificateFactory.getInstance("X509");
                    fis = new FileInputStream(chain);
                    try {
                        col = cf.generateCertificates(fis);
                    } catch (Exception e) {
                        fis.close();
                        cf = CertificateFactory.getInstance("X509", "IndigoSoftwareJCE");
                        fis = new FileInputStream(chain);
                        col = cf.generateCertificates(fis);
                    }
                }
            } catch (Exception e) {
                throw e;
            }
            for (Iterator<? extends Certificate> it = col.iterator(); it.hasNext(); ) {
                X509Certificate cert = (X509Certificate)it.next();
                list.add(cert);
            }
        } catch (Exception e) {
            throw e;
        } finally {
            if (fis != null) {
                fis.close();
            }
        }
        return list;
    }

    public boolean verifiyCertificateByKeyStore(X509Certificate cert, Collection<CRL> crls, Calendar calendar) {
        try {
            for (Enumeration<String> aliases = this.keyStore.aliases(); aliases.hasMoreElements();) {
                try {
                    String alias = aliases.nextElement();
                    if (!this.keyStore.isCertificateEntry(alias)) {
                        continue;
                    }
                    X509Certificate certStoreX509 = (X509Certificate)this.keyStore.getCertificate(alias);
                    if (verifyCert(certStoreX509, crls, calendar) != null) {
                        continue;
                    }
                    try {
                        cert.verify(certStoreX509.getPublicKey());
                        return true;
                    } catch (Exception e) {}
                } catch (Exception ex) {}
            }
        } catch (Exception e) {}
        return false;
    }

    public String verifyCert(X509Certificate cert, Collection<CRL> crls, Calendar calendar) {
        if (calendar == null) {
            calendar = new GregorianCalendar();
        }
        if (cert.hasUnsupportedCriticalExtension()) {
            for (String oid : cert.getCriticalExtensionOIDs()) {
                if ("2.5.29.15".equals(oid) && cert.getKeyUsage()[0]) {
                    continue;
                }
                try {
                    if ("2.5.29.37".equals(oid) && cert.getExtendedKeyUsage().contains("1.3.6.1.5.5.7.3.8")) {
                        continue;
                    }
                } catch (CertificateParsingException e) {
                }
                return "Has unsupported critical extension";
            }
        }
        try {
            cert.checkValidity(calendar.getTime());
        } catch (Exception e) {
            return e.getMessage();
        }
        if (crls != null) {
            for (CRL crl : crls) {
                if (crl.isRevoked(cert)) {
                    return "Certificate revoked";
                }
            }
        }
        return null;
    }

    public void verify(X509Certificate certificate, PublicKey publicKey) throws InvalidKeyException, CertificateException, NoSuchAlgorithmException, NoSuchProviderException, SignatureException {
        certificate.verify(publicKey, getProvider());
    }

    public KeyStore genKeyStore4Verify(SignatureDevice device, String chain) throws Exception {
        String provider = null;
        String keyStoreType = null;
        if (device != null) {
            provider = device.getProvider();
            keyStoreType = device.getParameters().get("keyStoreType");
        }
        KeyStore ks = null;
        if (keyStoreType == null) {
            keyStoreType = KeyStore.getDefaultType();
        }
        if (provider != null) {
            ks = KeyStore.getInstance(keyStoreType, provider);
        } else {
            ks = KeyStore.getInstance(keyStoreType);
        }
        ks.load(null, null);
        FileInputStream fis = null;
        try {
            File chainFile = new File(chain);
            if (!chainFile.exists()) {
                throw new Exception("Certificate chain file doesn't exist:" + chain);
            }
            fis = new FileInputStream(chain);
            CertificateFactory cf = null;
            if (provider != null) {
                cf = CertificateFactory.getInstance("X509", provider);
            } else {
                cf = CertificateFactory.getInstance("X509");
            }
            Exception error = null;
            Collection<? extends Certificate> col = null;
            try {
                col = cf.generateCertificates(fis);
            } catch (CertificateException e) {
                error = e;
            }
            try {
                if (col == null) {
                    IOUtils.closeQuietly(fis);
                    cf = CertificateFactory.getInstance("X509");
                    fis = new FileInputStream(chain);
                    col = cf.generateCertificates(fis);
                }
            } catch (Exception e) {
                throw e;
            }
            for (Iterator<? extends Certificate> it = col.iterator(); it.hasNext(); ) {
                X509Certificate cert = (X509Certificate)it.next();
                ks.setCertificateEntry(cert.getSerialNumber().toString(36), cert);
            }
        } catch (Exception e) {
            throw e;
        } finally {
            if (fis != null) {
                fis.close();
            }
        }
        return ks;
    }

    public Collection<Certificate> readCerts(byte[] contentsKey) throws Exception {
        return null;
    }

    public String getTrustCert() {
        return this.trustCert;
    }

    public void setTrustCert(String trustCert) {
        this.trustCert = trustCert;
    }

    public String getCertExtensionValue(Certificate certificate, String key) throws Exception {
        return null;
    }

    public void setProvider(String provider) {
        this.provider = provider;
    }
}

