/*
 * $Id:  $
 *
 * This file is part of the jcar (R) project.
 * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
 * Authors: laurent.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License
 * as published by the Free Software Foundation, either version 3 of
 * the License, or(at your option)any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Affero General Public License for more details.
 * You should have received a copy of the GNU Affero General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses>
 */
package com.indigosoftware.adapter.jce;

import com.indigosoftware.adapter.jce.common.SignatureDevice;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.security.InvalidKeyException;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

public class IndigoSoftwareJCEAdapter extends SignatureDevice {
    public static final Log logger = LogFactory.getLog(IndigoSoftwareJCEAdapter.class);

    private static volatile Boolean loaded = Boolean.valueOf(false);
    private PrivateKey key;

    public void initialize(Map<String, String> env)
            throws Exception {
        FileInputStream fis = null;
        try {
            String pkFile = (String) env.get("pkFile");
            if ((pkFile == null) || ("".equals(pkFile))) {
                Exception ie = new Exception("The parameters pkFile must be supplied if the mode is keystore");
                throw ie;
            }
            this.parameters.put("keyStoreFile", pkFile);

            String ksType = (String) env.get("ksType");
            if (ksType == null) {
                ksType = KeyStore.getDefaultType();
            }

            if (ksType.equals("IndigoSoftware")) {
                this.provider = "IndigoSoftwareJCE";
            }
            this.parameters.put("keyStoreType", ksType);

            String ksPWD = (String) env.get("ksPWD");
            if (ksPWD == null) {
              Exception ie = new Exception("The parameters ksPWD must be supplied if the mode is keystore");
                throw ie;
            }
            this.parameters.put("keyStorePWD", ksPWD);

            String pkName = (String) env.get("pkName");
            if ((pkName == null) || ("".equals(pkName))) {
              Exception ie = new Exception("The parameters pkName must be supplied if the mode is keystore");
                throw ie;
            }

            String pkPWD = (String) env.get("pkPWD");
            if (pkPWD == null) {
              Exception ie = new Exception("The parameters pkPWD must be supplied if the mode is keystore");
                throw ie;
            }

            if (!loaded.booleanValue()) {
                synchronized (loaded) {
                    if (!loaded.booleanValue()) {
                        Security.addProvider(new BouncyCastleProvider());
                        loaded = Boolean.valueOf(true);
                    }
                }

            }

            fis = new FileInputStream(pkFile);
            try {
                if (this.provider != null)
                    this.keyStore = KeyStore.getInstance(ksType, getProvider());
                else {
                    this.keyStore = KeyStore.getInstance(ksType);
                }
            } catch (Exception e) {
                throw e;
            }

            this.keyStore.load(fis, ksPWD.toCharArray());
            PrivateKey key = (PrivateKey) this.keyStore.getKey(pkName, pkPWD.toCharArray());
            if (key == null) {
                String message = pkName + " can't found ,keyStore listed as ";
                Enumeration en = this.keyStore.aliases();
                while (en.hasMoreElements()) {
                    message = message + (String) en.nextElement() + ";";
                }
                throw new Exception(message);
            }

            Certificate[] chain = this.keyStore.getCertificateChain(pkName);
            this.key = key;
            this.chain = chain;
        } catch (Exception e) {
          Exception ie = new Exception("IndigoSoftwareJCE initialize occur exception, errorMessage = [{message}]", e);
            logger.error(ie);
            throw ie;
        } finally {
            try {
                if (fis != null)
                    fis.close();
            } catch (IOException e) {
                fis = null;
            }
        }
    }

    public Signature getSignature(String algorithm) throws Exception {
        Signature sin = null;
        try {
            if (algorithm.equalsIgnoreCase("SM3WithSM2")) {
                sin = Signature.getInstance(algorithm, getProvider());
            } else {
                sin = Signature.getInstance(algorithm);
            }
        } catch (Exception e) {
            logger.warn(
                    "algorithm [{algorithm}] in JCE Provider[{provider}] is not implemented, try to call normal provider", e);
            sin = Signature.getInstance(algorithm, "Sun");
        }
        return sin;
    }

    public MessageDigest getMessageDigest(String hashAlgorithm) throws Exception {
        MessageDigest md = null;
        try {
            if (hashAlgorithm.equalsIgnoreCase("SM3")) {
                md = MessageDigest.getInstance(hashAlgorithm, getProvider());
            } else {
                md = MessageDigest.getInstance(hashAlgorithm);
            }
        } catch (Exception e) {
            logger.warn("hashAlgorithm [{hashAlgorithm}] in JCE Provider[{provider}] is not implemented, try to call normal provider", e);
            md = MessageDigest.getInstance(getHashAlgorithm(), "Sun");
        }
        return md;
    }

    public byte[] sign(byte[] originData)
            throws Exception {
        byte[] signedData = null;
        Signature signature = null;
        try {
            if (getProvider() != null)
                signature = Signature.getInstance(this.hashAlgorithm + "With" + this.encryptionAlgorithm, getProvider());
            else {
                signature = Signature.getInstance(this.hashAlgorithm + "With" + this.encryptionAlgorithm);
            }
            signature.initSign(this.key);
            signature.update(originData);
            signedData = signature.sign();
        } catch (Exception e) {
            Exception ie = new Exception("IndigoSoftwareJCE sign occur exception, errorMessage = [{message}]");
            logger.error("IndigoSoftwareJCE sign occur exception, errorMessage = [{message}]");
            throw ie;
        }
        return signedData;
    }

    public void destroy() {
    }

    public Object verify(byte[] data)
            throws Exception {
        return null;
    }

    public Collection<Certificate> readCerts(byte[] contentsKey)
            throws Exception {
        org.bouncycastle.jce.provider.X509CertParser cr = new org.bouncycastle.jce.provider.X509CertParser();
        cr.engineInit(new ByteArrayInputStream(contentsKey));
        return cr.engineReadAll();
    }

    public void verify(X509Certificate certificate, PublicKey publicKey) throws InvalidKeyException, CertificateException, NoSuchAlgorithmException, NoSuchProviderException, SignatureException {
        try {
            certificate.verify(publicKey, getProvider());
        } catch (Exception e) {
            certificate.verify(publicKey);
        }
    }
}