/*
  * $Id:  $
  *
  * This file is part of the jcar (R) project.
  * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
  * Authors: laurent.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of the GNU Affero General Public License
  * as published by the Free Software Foundation, either version 3 of
  * the License, or(at your option)any later version.
  *
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  * or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU Affero General Public License for more details.
  * You should have received a copy of the GNU Affero General Public License
  * along with this program; if not, see <http://www.gnu.org/licenses>
  */
package com.indigopacific.customagent;


import org.apache.log4j.Logger;

import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.TimerTask;
import java.util.concurrent.ExecutorService;


public class PdfSignatureScanTask extends TimerTask {

    private static Logger logger = Logger.getLogger(PdfSignatureScanTask.class);


    String scanDir;
    String targetDir;
    ExecutorService agent_threadpool;

    public PdfSignatureScanTask(String scanDir, String targetDir, ExecutorService agent_threadpool) {
        this.scanDir = scanDir;
        this.targetDir = targetDir;
        this.agent_threadpool = agent_threadpool;
    }
    private static boolean isValidPdf(Path file) {
        return Files.isRegularFile(file) &&
                file.getFileName().toString().toLowerCase().endsWith(".pdf");
    }


    @Override
    public void run() {
        try {
            if (!Files.exists(Paths.get(targetDir))) {
                Files.createDirectories(Paths.get(targetDir));
            }
            logger.info("\n开始执行扫描任务...scanDir=" + scanDir + " targetDir=" + targetDir);
            long startTime = System.currentTimeMillis();

            // 执行扫描复制操作
            if (this.scanDir != null && !this.scanDir.equals("")) {
                Path sourceDir = Paths.get(this.scanDir);

                // 只遍历第一层子目录
                DirectoryStream<Path> directories = Files.newDirectoryStream(sourceDir, new DirectoryStream.Filter<Path>() {
                    @Override
                    public boolean accept(Path entry) throws IOException {
                        return Files.isDirectory(entry);
                    }
                });

                try {
                    for (Path businessDir : directories) {
                        final String businessParam = businessDir.getFileName().toString();
                        logger.info("处理业务目录: " + businessParam);
                        if (!Files.exists(Paths.get(targetDir, businessParam))) {
                            Files.createDirectories(Paths.get(targetDir, businessParam));
                        }
                        // 遍历业务目录下的PDF文件
                        Files.walkFileTree(businessDir, new SimpleFileVisitor<Path>() {
                            @Override
                            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                                // 过滤PDF文件
                                if (isValidPdf(file)) {
                                    try {
                                        //加载文件标识
                                        Path loadPath = file.resolveSibling(file.getFileName().toString() + ".load");
                                        Files.move(file, loadPath, StandardCopyOption.ATOMIC_MOVE);


                                        // 构造目标文件路径，包含业务参数
                                        Path targetPath = Paths.get(targetDir, businessParam, "out-" + file.getFileName().toString());
                                        String[] paramArrayOfString1 = new String[]{
                                                loadPath.toString(),
                                                targetPath.toString(),
                                                businessParam  // 传递业务参数给SignatureAgent
                                        };
                                        logger.info("调用agent处理PDF: " + file.toString() + ", 业务参数: " + businessParam);

                                        SignatureAgent signatureAgent = new SignatureAgent(); // 在外部创建，避免重复实例化

                                        Agent agentinfo = new Agent();
                                        agentinfo.setAgentEx(signatureAgent);

                                        agentinfo.setArgs(paramArrayOfString1);
                                        agentinfo.setMsg(new Message());
                                        agent_threadpool.submit(new AgentExecutor(agentinfo, loadPath));
                                    } catch (Exception e) {
                                        logger.error("执行失败 [" + file + "]: " + e.getMessage(), e);
                                    }
                                }
                                return FileVisitResult.CONTINUE;
                            }

                            @Override
                            public FileVisitResult visitFileFailed(Path file, IOException exc) {
                                logger.warn("无法访问文件: " + file);
                                return FileVisitResult.CONTINUE;
                            }
                        });
                    }
                } finally {
                    directories.close();
                }
            }

            long duration = System.currentTimeMillis() - startTime;
            System.out.printf("扫描完成，耗时%d毫秒%n", duration);
        } catch (Exception e) {
            logger.error("扫描任务执行失败: " + e.getMessage(), e);
        }
    }
}