 /*
  * $Id:  $
  *
  * This file is part of the jcar (R) project.
  * Copyright (c) 2014-2018 北京益高亚太信息技术有限公司
  * Authors: laurent.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of the GNU Affero General Public License
  * as published by the Free Software Foundation, either version 3 of
  * the License, or(at your option)any later version.
  *
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  * or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU Affero General Public License for more details.
  * You should have received a copy of the GNU Affero General Public License
  * along with this program; if not, see <http://www.gnu.org/licenses>
  */
package com.indigopacific.customagent;

import com.itextpdf.text.pdf.*;

import java.io.FileOutputStream;

/**
 * PDF旋转Agent
 * 支持旋转90度、180度、270度
 * **/
public class PdfRotatorAgent implements CustomAgentEx{
    @Override
    public void execute(String[] paramArrayOfString) throws Exception {
        String pdfPath = paramArrayOfString[0];
        Integer rotationAngle = Integer.valueOf(paramArrayOfString[1]);
        String outPath = paramArrayOfString[2];
        rotatePdf(pdfPath,outPath,rotationAngle);
    }

    @Override
    public String getVersion() {
        return "v1.6.0";
    }


    /**
     * 旋转PDF所有页面
     * @param inputPath  输入文件路径
     * @param outputPath 输出文件路径
     * @param rotation   旋转角度（必须是90的倍数）
     * @throws Exception
     */
    public static void rotatePdf(String inputPath, String outputPath, int rotation) throws Exception {
        validateRotation(rotation);

        PdfReader reader = null;
        PdfStamper stamper = null;
        try {
            reader = new PdfReader(inputPath);
            stamper = new PdfStamper(reader, new FileOutputStream(outputPath));

            int pages = reader.getNumberOfPages();
            for (int i = 1; i <= pages; i++) {
                PdfDictionary pageDict = reader.getPageN(i);
                applyRotation(pageDict, rotation);
            }
        } finally {
            if (stamper != null) {
                stamper.close();
            }
            if (reader != null) {
                reader.close();
            }
        }
    }

    /**
     * 旋转指定页面
     * @param inputPath   输入文件路径
     * @param outputPath  输出文件路径
     * @param rotation    旋转角度
     * @param pageNumbers 目标页码（从1开始）
     * @throws Exception
     */
    public static void rotateSpecificPages(String inputPath, String outputPath,
                                           int rotation, int... pageNumbers) throws Exception {
        validateRotation(rotation);

        PdfReader reader = null;
        PdfStamper stamper = null;
        try {
            reader = new PdfReader(inputPath);
            stamper = new PdfStamper(reader, new FileOutputStream(outputPath));

            for (int pageNum : pageNumbers) {
                if (pageNum < 1 || pageNum > reader.getNumberOfPages()) {
                    continue;
                }
                PdfDictionary pageDict = reader.getPageN(pageNum);
                applyRotation(pageDict, rotation);
            }
        } finally {
            if (stamper != null) {
                stamper.close();
            }
            if (reader != null) {
                reader.close();
            }
        }
    }

    private static void validateRotation(int rotation) {
        if (rotation % 90 != 0) {
            throw new IllegalArgumentException("Rotation must be multiple of 90 degrees");
        }
        // 标准化角度到0-359
        rotation = (rotation % 360 + 360) % 360;
    }

    private static void applyRotation(PdfDictionary pageDict, int additionalRotation) {
        // 获取原始旋转角度
        int originalRotation = 0;
        PdfNumber rotate = pageDict.getAsNumber(PdfName.ROTATE);
        if (rotate != null) {
            originalRotation = rotate.intValue();
        }

        // 计算新角度
        int newRotation = (originalRotation + additionalRotation) % 360;
        newRotation = (newRotation + 360) % 360; // 确保正值

        // 更新页面旋转属性
        pageDict.put(PdfName.ROTATE, new PdfNumber(newRotation));
    }

    public static void main(String[] args) {
        try {
            // 示例：全部页面旋转90度
            rotatePdf("input.pdf", "output-all.pdf", 90);
            // 示例：旋转第2页和第4页180度
            rotateSpecificPages("input.pdf", "output-spec.pdf", 180, 2, 4);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
